<?php
namespace App\Http\Controllers;

use App\Models\Order;
use App\Models\Ticket;
use Illuminate\Support\Facades\DB;

class DashboardController extends Controller
{
    /**
     * @var bool
     */
    private bool $api_call;
    /**
     * @brief assign true|false to $api_call by checking if the request is an API call
     *
     * @return void
     */
    public function __construct() {
        $this->api_call = request()->header('Accept') == "application/json";
    }
    /**
     * @brief fetch dashboard content as array
     *
     * @return \Illuminate\Contracts\View\View|string
     */
    public function viewProperDashboard() {
        $user = auth()->user();
        /**
         * fetch dashboard content based on user role
         * 10: admin
         * 30: user | customer
         */
        switch ($user->role) {
            case 10:
                // throw a 403 error if the request is an API call
                if ($this->api_call) {
                    auth()->logout();
                    throwForbiddenError(['errors' => ['Admin is forbidden to access this content via the API']]);
                    exit();
                }
                // fetching tickets summary
                $tickets = [
                    'sum'      => Ticket::all()->count(),
                    'answered' => Ticket::where('status', 0)->where('seen_by_user', 0)->count(),
                    'closed'   => Ticket::where('status', 1)->count(),
                    'awaiting' => Ticket::where('status', 0)->where('seen_by_operator', 0)->count()
                ];
                // fetching orders summary
                $orders['sum']       = count(Order::all());
                $orders['done']      = count(Order::where('status', '=', 'done')->get());
                Order::where('status', '=', 'reserved')->where('placement_datetime', '<=', date('Y-m-d H:i:s', strtotime("-15 minutes")))->update(['status'=>'cancelled']);
                $orders['reserved']  = count(Order::where('status', '=', 'reserved')->get());
                $orders['cancelled'] = count(Order::where('status', '=', 'cancelled')->get());
                $order = ["sum"=>$orders['sum'], "done"=>$orders['done'], "cancelled"=>$orders['cancelled'], "reserved"=>$orders['reserved']];
                // returning the proper view
                return view('v1/02_dash-template/admin-pages/dashboard-admin', ['user'=>$user, 'tickets'=>$tickets, "order"=>$order]);
            case 30:
                return !$this->api_call ?
                    view('v1/02_dash-template/customer-pages/dashboard-customer', ['user'=>$user]) :
                    successfulResponse(['user'=>$user]);
            default:
                !$this->api_call ? abort(401) : throwUnauthorizedError();
        }
    }
}
