<?php

namespace App\Http\Controllers;

use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Validation\ValidationException;
use function auth;

class UsersController extends Controller
{
    public function index() {
        return isAdmin() ?
            view('v1/02_dash-template/admin-pages/edit-user-info', ['user'=>auth()->user()]) :
            view('v1/02_dash-template/customer-pages/edit-user-info', ['user'=>auth()->user()]);
    }
    public function getUsersList() {
        if (!isAdmin())
            abort(403);
        return view("v1/02_dash-template/admin-pages/users-list", ['users'=>User::where('id', '<>', auth()->user()->id)->get()]);
    }
    public function requestHandler(): \Illuminate\Http\RedirectResponse {
        if (request()->has('action')) {
            switch (request('action')) {
                case 'edit_password':
                    $this->editPassword();
                    return back()->with('msg', 'ویرایش رمز عبور با موفقیت انجام شد');
                case 'change_user_data':
                    $this->changeUserParams();
                    return back()->with('msg', 'تغییرات با موفقیت اعمال شد');
                default:
                    throw ValidationException::withMessages(['err' => 'درخواست نامعتبر']);
            }
        } else
            throw ValidationException::withMessages(['err' => 'درخواست نامعتبر']);
    }

    /**
     * @throws ValidationException
     */
    private function editPassword() {
        if (request()->has('old_password')) {
            $attributes = [
                'email'=>auth()->user()->email,
                'password'=>request('old_password')
            ];
            if (auth()->attempt($attributes)) {
                request()->validate([
                    'password' => [
                        'required',
                        'between:8,100',
                        'regex:/^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)[a-zA-Z\d]{8,}$/',
                        'confirmed'
                    ],
                    'password_confirmation'=>'required'
                ],[
                    'password.required' => 'فیلد رمز عبور الزامی می‌باشد',
                    'password.regex' => 'طول رمز عبور باید حداقل ۸ کاراکتر و شامل اعداد، حروف کوچک و بزرگ باشد',
                    'password.between' => 'طول رمز عبور باید بین ۸ تا ۱۰۰ کاراکتر باشد',
                    'password.confirmed' => 'فیلد‌های رمز عبور جدید و تکرار رمز جدید، یکسان نیستند',
                    'password_confirmation.required' => 'فیلد تکرار رمز عبور الزامی می‌باشد'
                ]);
                User::where('id', '=', auth()->user()->id)->update(['password' => bcrypt(request('password'))]);
            } else
                throw ValidationException::withMessages(['err' => 'رمز عبور صحیح نیست']);
        } else
            throw ValidationException::withMessages(['err' => 'فیلد رمز عبور فعلی الزامیست']);
    }

    private function changeUserParams() {
        request()->validate([
            'uid'=>'required|numeric|exists:users,id',
            'role'=>'required|numeric',
            'credit'=>'required|numeric',
        ],[
            'uid.exists' => 'کاربر نامعتبر',
            'uid.numeric' => 'کاربر نامعتبر',
            'uid.required' => 'کاربر نامعتبر',
            'role.required' => 'نقش نامعتبر',
            'role.numeric' => 'نقش نامعتبر',
            'credit.required' => 'اعتبار نامعتبر',
            'credit.numeric' => 'اعتبار نامعتبر',
        ]);
        User::where('id', '=', request('uid'))->update(['role' => request('role'), 'credit' => request('credit')]);
    }
}
